import os
from PIL import Image

def get_average_rgb(image_path):
    """
    Calculates the average RGB color of an image.

    Args:
        image_path (str): The path to the image file.

    Returns:
        tuple: A tuple (r, g, b) representing the average RGB color,
               or None if the image cannot be processed.
    """
    try:
        with Image.open(image_path) as img:
            # Convert to RGB if not already
            if img.mode != 'RGB':
                img = img.convert('RGB')

            # Get image data as a list of pixels
            pixels = list(img.getdata())

            # Calculate sum of R, G, B components
            r_sum = 0
            g_sum = 0
            b_sum = 0
            for pixel in pixels:
                r_sum += pixel[0]
                g_sum += pixel[1]
                b_sum += pixel[2]

            # Calculate average
            num_pixels = len(pixels)
            avg_r = r_sum / num_pixels
            avg_g = g_sum / num_pixels
            avg_b = b_sum / num_pixels

            return (avg_r, avg_g, avg_b)

    except Exception as e:
        print(f"Error processing image {image_path}: {e}")
        return None

def main():
    """
    Main function to process images in the current directory and generate a text file
    with normalized RGB (Vector3f) values.
    """
    # Get the current working directory
    directory_path = os.getcwd()

    output_filename = "fluid_colors.txt"
    image_extensions = ('.png', '.jpg', '.jpeg', '.bmp', '.gif') # Add or remove as needed

    print(f"\nProcessing images in the current directory: {directory_path}")
    print(f"Output will be saved to: {os.path.join(directory_path, output_filename)}\n")

    with open(output_filename, 'w') as f:
        for filename in os.listdir(directory_path):
            file_path = os.path.join(directory_path, filename)

            if os.path.isfile(file_path) and filename.lower().endswith(image_extensions):
                print(f"  Processing {filename}...")
                avg_rgb = get_average_rgb(file_path)

                if avg_rgb:
                    # Convert average RGB to integers
                    int_r = int(avg_rgb[0])
                    int_g = int(avg_rgb[1])
                    int_b = int(avg_rgb[2])

                    # Format as hex string
                    hex_color = f"#{int_r:02x}{int_g:02x}{int_b:02x}"
                    f.write(f"// {filename}\n")
                    f.write(f'"{hex_color}",\n\n') # Added a comma and extra newline for readability/copy-paste and quotes for hex string

                    print(f"    - Average RGB: ({int_r}, {int_g}, {int_b}) -> {hex_color}")
                else:
                    print(f"    - Could not process {filename}. Skipping.")
            # else: # Optional: Uncomment if you want to explicitly see skipped files
            #    print(f"  Skipping non-image file/folder: {filename}")

    print(f"\nProcessing complete. Check '{output_filename}' in the current directory for results.")

if __name__ == "__main__":
    main()